// --- Code dibuat oleh Teguh Sakura ( teguh.sakura@yahoo.com / toko.teguh.sakura@gmail.com ) ---

// --- WiFi AP Hotspot ---
#include <WiFi.h>
const char* ap_ssid = "Teguh_Sakura";
const char* ap_password = "";

// --- Web Server dan Web Html ---
#include <WebServer.h>
#include "index.h"
#include <ArduinoJson.h>

// --- Sensor Suhu dan Sensor Kelembaban ---
#include <OneWire.h>
#include <DallasTemperature.h>
const int oneWireBus = 32;
const int soilMoisturePin = 34;

// --- OLED Display ---
#include <Wire.h>
#include <Adafruit_GFX.h>
#include <Adafruit_SSD1306.h>
#define SCREEN_WIDTH 128
#define SCREEN_HEIGHT 64
#define OLED_RESET -1
Adafruit_SSD1306 display(SCREEN_WIDTH, SCREEN_HEIGHT, &Wire, OLED_RESET);

// --- Sensor Suhu & Port Web Server ---
OneWire oneWire(oneWireBus);
DallasTemperature sensors(&oneWire);
WebServer server(80);

// --- Kalibrasi Sensor Kelembaban pakai nilai raw analog ---
int KondisiKering = 2600;
int KondisiBasah = 1300;

// --- RS485 PH Sensor ---
const byte ph_soil_reg[8] = {0x01, 0x03, 0x00, 0x00, 0x00, 0x01, 0x84, 0x0A};
#define RESPONSE_LENGTH 7

uint16_t calculateCRC(byte *array, uint8_t length) {
  uint16_t crc = 0xFFFF;
  for (uint8_t i = 0; i < length; i++) {
    crc ^= array[i];
    for (uint8_t j = 0; j < 8; j++) {
      if ((crc & 0x0001) != 0) {
        crc >>= 1;
        crc ^= 0xA001;
      } else {
        crc >>= 1;
      }
    }
  }
  return crc;
}


// --- Timer buat ambil data serta tampilkan ke OLED ---
unsigned long previousLogMillis = 0;
unsigned long previousDisplayMillis = 0;
const long displayInterval = 3000; // 3 detik

// --- Nilai Sensor saat ini ---
float currentTemperatureC = 0.0;
int currentMoisturePercent = 0;
float soil_ph = 0.0;

// --- Fungsi Prototipe ---
void bacaSensor();
void Oled();
void handleRoot();
void handleData();


void setup() {
    Serial.begin(115200);
    Serial.println("\n\n--- Alat ukur Suhu, Kelembaban dan pH Tanah pakai ESP32 ---");

    // --- OLED ---
    if(!display.begin(SSD1306_SWITCHCAPVCC, 0x3C)) {
        Serial.println(F("SSD1306 gagal di deteksi")); for(;;);
    }
    display.clearDisplay();
    display.setTextSize(1);
    display.setTextColor(SSD1306_WHITE);
    display.setRotation(2);
    display.setCursor(0, 0);
    display.println("Memulai...");
    display.display();

    // --- Sensor ---
    sensors.begin();
    Serial2.begin(4800, SERIAL_8N1, 16, 17); // RX2 (GPIO16), TX2 (GPIO17)

    // --- Wi-Fi Access Point ---
    WiFi.softAP(ap_ssid, ap_password);
    IPAddress myIP = WiFi.softAPIP();
    Serial.print("Masuk mode AP Hotspot. IP Address: ");
    Serial.println(myIP);

    display.clearDisplay();
    display.setTextColor(SSD1306_WHITE);
    display.setCursor(0, 0);
    display.setTextSize(2);
    display.println("WiFi AP:");
    display.setTextSize(1);
    display.println(ap_ssid);
    display.println(ap_password);
    display.display();
    delay(2000);

    // --- Web Server ---
    server.on("/", HTTP_GET, handleRoot);
    server.on("/data", HTTP_GET, handleData);

    server.begin();
    Serial.println("Web Server dijalankan.");

    // --- Baca sensor dan tampilkan di OLED ---
    bacaSensor();
    Oled();
}

void loop() {
    server.handleClient();
    unsigned long currentMillis = millis();

    // Baca Sensor dan Update tampilan OLED setiap 'displayInterval'
    if (currentMillis - previousDisplayMillis >= displayInterval) {
        previousDisplayMillis = currentMillis;
        bacaSensor();
        Oled();
    }
}

// --- Cek Sensor ---
void bacaSensor() {
  // Baca temperatur dari DS18B20
  sensors.requestTemperatures();
  float tempC = sensors.getTempCByIndex(0);
  if (tempC != DEVICE_DISCONNECTED_C) {
      currentTemperatureC = tempC;
  } else {
      currentTemperatureC = -999.0; // Indikasi gangguan
  }
  // Baca Kelembaban lewat analog ( adc )
  int rawMoisture = analogRead(soilMoisturePin);
  // Map nilai raw analog ke persen, dibatasi antara 0 sampai 100
  currentMoisturePercent = map(rawMoisture, KondisiKering, KondisiBasah, 0, 100);
  currentMoisturePercent = constrain(currentMoisturePercent, 0, 100);

  // Baca sensor pH lewat RS485
  byte response[RESPONSE_LENGTH];
  int bytesRead = 0;

  // Bersihkan semua data masuk di Serial2 sebelum meminta data baru ke sensor pH
  while (Serial2.available()) {
    Serial2.read();
  }

  Serial2.write(ph_soil_reg, sizeof(ph_soil_reg));
  Serial2.flush();

  unsigned long startTime = millis();
  while (bytesRead < RESPONSE_LENGTH && (millis() - startTime < 500)) { // 500ms timeout
    if (Serial2.available()) {
      response[bytesRead++] = Serial2.read();
    }
  }

  if (bytesRead == RESPONSE_LENGTH) {
    uint16_t receivedCRC = (response[RESPONSE_LENGTH - 1] << 8) | response[RESPONSE_LENGTH - 2];
    uint16_t calculatedCRC = calculateCRC(response, RESPONSE_LENGTH - 2);

    if (receivedCRC == calculatedCRC) {
      uint16_t rawPHValue = (response[3] << 8) | response[4];
      soil_ph = rawPHValue / 10.0;

      Serial.printf("Nilai Raw PH: ");
      Serial.print(rawPHValue);
      Serial.printf(" -> PH: ");
      Serial.println(soil_ph, 2); // Print pH with 2 decimal places
    } else {
      Serial.printf("CRC tidak sama! Diterima: 0x");
      Serial.print(receivedCRC, HEX);
      Serial.printf(", Kalkulasi: 0x");
      Serial.println(calculatedCRC, HEX);
      Serial.printf("Respons: ");
      for (int i = 0; i < bytesRead; i++) {
        Serial.print("0x");
        if (response[i] < 0x10) Serial.print("0");
        Serial.print(response[i], HEX);
        Serial.print(" ");
      }
      Serial.println();
    }
  } else {
    Serial.printf("Tidak lengkap atau gak ada respon. membaca bit: ");
    Serial.println(bytesRead);
    Serial.printf("Respon: ");
    for (int i = 0; i < bytesRead; i++) {
      Serial.printf("0x");
      if (response[i] < 0x10) Serial.printf("0");
      Serial.print(response[i], HEX);
      Serial.print(" ");
    }
    Serial.println();
  }
}

// --- Cek Tanaman ---
String cekTanaman(float suhu, int lembab, float ph) {
  if (suhu >= 17.0 && suhu <= 28.0 &&
      lembab >= 40 && lembab <= 60 &&
      ph >= 6.0 && ph <= 7.0) {
    return "Bayam";
  }
  else if (suhu >= 24.0 && suhu <= 28.0 &&
           lembab >= 60 && lembab <= 80 &&
           ph >= 5.5 && ph <= 7.0) {
    return "Cabe";
  }
  else if (suhu >= 25.0 && suhu <= 30.0 &&
           lembab >= 80 && lembab <= 100 &&
           ph >= 5.5 && ph <= 7.0) {
    return "Kangkung";
  }
  else if (suhu >= 25.0 && suhu <= 30.0 &&
           lembab >= 70 && lembab <= 80 &&
           ph >= 6.0 && ph <= 7.2) {
    return "Melon";
  }
  else {
    return "x x x";
  }
}

// --- Tampilkan ke OLED ---
void Oled() {
  String rekomendasiTanaman = cekTanaman(currentTemperatureC, currentMoisturePercent, soil_ph);

  display.clearDisplay();

  display.setTextSize(1);
  display.setTextColor(SSD1306_WHITE);

  display.setCursor(0, 0);
  display.setTextSize(1);
  display.print("Tanaman: ");
  display.setTextSize(2);
  display.println(rekomendasiTanaman);

  display.setCursor(0, 16);
  display.setTextSize(1);
  display.print("Suhu: ");
  display.setTextSize(2);
  display.print(currentTemperatureC, 1);
  display.println(" C");

  display.setCursor(0, 32);
  display.setTextSize(1);
  display.print("Kelembaban: ");
  display.setTextSize(2);
  display.print(currentMoisturePercent);
  display.println("%");

  display.setCursor(0, 48);
  display.setTextSize(1);
  display.print("pH: ");
  display.setTextSize(2);
  display.print(soil_ph, 1);
  display.println("");

  display.display();
}

// --- Buat Web Html ---
void handleRoot() {
  server.send(200, "text/html", WEB_HTML);
}

void handleData() {
  bacaSensor();
  String rekomendasiTanaman = cekTanaman(currentTemperatureC, currentMoisturePercent, soil_ph);
  StaticJsonDocument<200> doc; // Allocate a fixed-size JSON document (adjust size if needed)
  doc["temperature"] = currentTemperatureC;
  doc["moisture"] = currentMoisturePercent;
  doc["ph"] = soil_ph;
  doc["plant"] = rekomendasiTanaman;

  String jsonString;
  serializeJson(doc, jsonString); // Convert JSON document to string

  server.send(200, "application/json", jsonString);
}
